from flask import Flask, request, jsonify
from flask_cors import CORS
import sqlite3, os

DB_PATH = os.environ.get("DB_PATH", "data.db")

def get_db():
    conn = sqlite3.connect(DB_PATH)
    conn.row_factory = sqlite3.Row
    return conn

def init_db():
    conn = get_db()
    c = conn.cursor()
    c.execute("CREATE TABLE IF NOT EXISTS reviews(id TEXT PRIMARY KEY, title TEXT, author TEXT, blurb TEXT, genre TEXT, rating INTEGER)")
    c.execute("CREATE TABLE IF NOT EXISTS practice(id TEXT PRIMARY KEY, week TEXT, hours REAL)")
    c.execute("CREATE TABLE IF NOT EXISTS notes(id TEXT PRIMARY KEY, date TEXT, text TEXT)")
    c.execute("CREATE TABLE IF NOT EXISTS coach(id TEXT PRIMARY KEY, date TEXT, text TEXT)")
    c.execute("CREATE TABLE IF NOT EXISTS coach_tags(note_id TEXT, tag TEXT)")
    conn.commit(); conn.close()

app = Flask(__name__)
CORS(app)
init_db()

@app.get('/api/ping')
def ping(): return jsonify({"ok": True})

@app.get('/api/reviews')
def get_reviews():
    conn = get_db(); rows = conn.execute("SELECT * FROM reviews ORDER BY rowid DESC").fetchall(); conn.close()
    return jsonify([dict(r) for r in rows])

@app.post('/api/reviews')
def add_review():
    data = request.json
    conn = get_db()
    conn.execute("INSERT OR REPLACE INTO reviews(id,title,author,blurb,genre,rating) VALUES(?,?,?,?,?,?)",
                 (data['id'], data['title'], data.get('author',''), data.get('blurb',''), data.get('genre','fantasy'), int(data.get('rating',3))))
    conn.commit(); conn.close()
    return jsonify({"ok": True})

@app.get('/api/practice')
def get_practice():
    conn = get_db(); rows = conn.execute("SELECT * FROM practice ORDER BY week ASC").fetchall(); conn.close()
    return jsonify([dict(r) for r in rows])

@app.post('/api/practice')
def add_practice():
    d = request.json
    conn = get_db()
    conn.execute("INSERT OR REPLACE INTO practice(id,week,hours) VALUES(?,?,?)", (d['id'], d['week'], float(d.get('hours',0))))
    conn.commit(); conn.close()
    return jsonify({"ok": True})

@app.delete('/api/practice/<id>')
def del_practice(id):
    conn = get_db(); conn.execute("DELETE FROM practice WHERE id=?", (id,)); conn.commit(); conn.close()
    return jsonify({"ok": True})

@app.get('/api/notes')
def get_notes():
    conn = get_db(); rows = conn.execute("SELECT * FROM notes ORDER BY rowid DESC").fetchall(); conn.close()
    return jsonify([dict(r) for r in rows])

@app.post('/api/notes')
def add_note():
    d = request.json
    conn = get_db(); conn.execute("INSERT OR REPLACE INTO notes(id,date,text) VALUES(?,?,?)", (d['id'], d.get('date',''), d.get('text','')))
    conn.commit(); conn.close()
    return jsonify({"ok": True})

@app.get('/api/coach')
def get_coach():
    conn = get_db()
    notes = [dict(r) for r in conn.execute("SELECT * FROM coach ORDER BY rowid DESC")]
    for n in notes:
        tags = [r['tag'] for r in conn.execute("SELECT tag FROM coach_tags WHERE note_id=?", (n['id'],))]
        n['tags'] = tags
    conn.close()
    return jsonify(notes)

@app.post('/api/coach')
def add_coach():
    d = request.json
    conn = get_db()
    conn.execute("INSERT OR REPLACE INTO coach(id,date,text) VALUES(?,?,?)", (d['id'], d.get('date',''), d.get('text','')))
    conn.execute("DELETE FROM coach_tags WHERE note_id=?", (d['id'],))
    for t in d.get('tags', []):
        conn.execute("INSERT INTO coach_tags(note_id, tag) VALUES(?,?)", (d['id'], t))
    conn.commit(); conn.close()
    return jsonify({"ok": True})

if __name__ == '__main__':
    app.run(host='0.0.0.0', port=5000)
